#!/bin/bash

set -e -o pipefail

BASE_DIR=$(dirname "$0")
readonly BASE_DIR
readonly CONFIGURATION_DIR=$BASE_DIR/configuration
readonly DOCKER_COMPOSE_DIR=$BASE_DIR/docker-compose
readonly LICENSE_KEY_FILE=$CONFIGURATION_DIR/license-key.txt

production_enabled=false
reverse_proxy_enabled=false
while [ "$1" != "" ]; do
  case $1 in
    --production)
      production_enabled=true
      ;;
    --reverse-proxy)
      reverse_proxy_enabled=true
      ;;
    *)
      echo "ERROR: Unknown argument '$1'"
      exit 1
  esac
  shift
done

if [ "$production_enabled" = "true" ]; then
  readonly ENV_FILE=$CONFIGURATION_DIR/.env.production
else
  readonly ENV_FILE=$CONFIGURATION_DIR/.env.demo
fi

function is_valid_license() {
  grep -q "CAMUNDA LICENSE KEY" "$LICENSE_KEY_FILE" && grep -q "cawemo = true" "$LICENSE_KEY_FILE"
}

function is_config_variable_missing() {
  ! grep -q "$1" "$ENV_FILE"
}

function is_config_variable_empty() {
  grep -q "^$1=\s*$" "$ENV_FILE"
}

function get_config_variable() {
  sed -rn "s/^$1=(.*)$/\1/p" "$ENV_FILE"
}

function set_config_variable() {
  sed -i.bak "s/$1=/$1=$2/g" "$ENV_FILE"
}

function get_docs_version() {
  get_config_variable "CAWEMO_VERSION" "$ENV_FILE" | sed -r "s/\.[^.]*$//"
}

function generate_random_string() {
  LC_ALL=C tr -dc "a-zA-Z0-9" < /dev/urandom | fold -w 32 | head -n 1
}

function generate_jwk() {
  docker run --rm -t "registry.camunda.cloud/iam-ee/iam-utility:$(get_config_variable 'IAM_VERSION')" \
    yarn --silent run generate-jwk | tail -n 1 | sed -r "s/^.*(\{.*\}).*$/\1/"
}

function all_containers_healthy() {
  local not_healthy_count
  not_healthy_count=$(docker inspect --format="{{.State.Health.Status}}" "${CONTAINER_NAMES[@]}" | sed -r "/^healthy$/d" | wc -l)
  [ "$not_healthy_count" -eq 0 ]
}

if ! is_valid_license; then
  echo ""
  echo "ERROR: $LICENSE_KEY_FILE does not contain a valid Cawemo license."
  echo ""
  exit 1
fi

echo "Checking configuration file $ENV_FILE ..."

# shellcheck disable=SC2207
missing_variables=($(sed -rn "/(IAM_LDAP_|_SECRET|_KEY|SERVER_TLS_CERTIFICATES_PATH|IAM_SERVER_HOST)/!s/^(.*)=\s*$/\1/p" "$ENV_FILE"))
if [ ${#missing_variables[@]} -gt 0 ]; then
  echo ""
  echo "ERROR: Invalid configuration."
  echo ""
  echo "The following required environment variables are not set:"
  for var in "${missing_variables[@]}"; do
    echo " - $var"
  done
  echo ""
  echo "For a description of the different configuration settings please refer to:"
  echo "https://docs.camunda.org/cawemo/$(get_docs_version)/technical-guide/installation/configuration"
  echo ""
  exit 1
fi

if grep -q "LDAP_ENABLED=true" "$ENV_FILE"; then
  # shellcheck disable=SC2207
  missing_variables=($(sed -rn "s/^(IAM_LDAP_SERVER_URL|IAM_LDAP_UUID_ATTRIBUTE|IAM_LDAP_USER_NAME_ATTRIBUTES|IAM_LDAP_USER_EMAIL_ATTRIBUTE)=\s*$/\1/p" "$ENV_FILE"))
  if [ ${#missing_variables[@]} -gt 0 ]; then
    echo ""
    echo "ERROR: Invalid LDAP configuration."
    echo ""
    echo "The LDAP integration is enabled but the following required environment variables are not set:"
    for var in "${missing_variables[@]}"; do
      echo " - $var"
    done
    echo ""
    echo "Depending on your LDAP server, other settings might be required as well. For more details please refer to:"
    echo "https://docs.camunda.org/cawemo/$(get_docs_version)/technical-guide/installation/ldap-support"
    echo ""
    exit 1
  fi
fi

if [ $reverse_proxy_enabled = "true" ]; then
  missing_variables=()
  for variable_name in "SERVER_TLS_CERTIFICATES_PATH" "IAM_SERVER_HOST"; do
    if is_config_variable_missing "$variable_name" || is_config_variable_empty "$variable_name"; then
      missing_variables+=("$variable_name")
    fi
  done
  if [ ${#missing_variables[@]} -gt 0 ]; then
    echo ""
    echo "ERROR: Invalid reverse proxy configuration."
    echo ""
    echo "The reverse proxy is enabled but the following required environment variables are not set:"
    for var in "${missing_variables[@]}"; do
      echo " - $var"
    done
    echo ""
    echo "For more details please refer to:"
    echo "https://docs.camunda.org/cawemo/$(get_docs_version)/technical-guide/installation/reverse-proxy-configuration"
    echo ""
    exit 1
  fi
fi

for variable_name in "SERVER_SESSION_COOKIE_SECRET" "WEBSOCKET_SECRET" "CLIENT_SECRET" "IAM_DATABASE_ENCRYPTION_KEY"; do
  if is_config_variable_empty "$variable_name"; then
    echo "Generating random value for $variable_name"
    set_config_variable "$variable_name" "$(generate_random_string)"
  fi
done
if is_config_variable_empty "IAM_TOKEN_SIGNING_KEY"; then
  echo "Generating JWK for IAM_TOKEN_SIGNING_KEY"
  set_config_variable "IAM_TOKEN_SIGNING_KEY" "$(generate_jwk)"
fi
rm -f "${ENV_FILE}.bak"

echo "Starting Cawemo with docker compose ..."
echo ""
docker_compose_files=(-f "$DOCKER_COMPOSE_DIR"/docker-compose.yml)
if [ "$production_enabled" = "false" ]; then
  docker_compose_files+=(-f "$DOCKER_COMPOSE_DIR"/docker-compose.demo.yml)
fi
if [ "$reverse_proxy_enabled" = "true" ]; then
  docker_compose_files+=(-f "$DOCKER_COMPOSE_DIR"/docker-compose.reverse-proxy.yml)
fi
docker compose "${docker_compose_files[@]}" --env-file "$ENV_FILE" up -d

echo ""
echo "Waiting for Cawemo to be ready ..."

# shellcheck disable=SC2207
readonly CONTAINER_NAMES=($(docker compose "${docker_compose_files[@]}" --env-file "$ENV_FILE" config | sed -rn "s/container_name: (.*)$/\1/p"))
readonly TIMEOUT_SECONDS=120
readonly SLEEP_SECONDS=5
declare -i waited=0
until all_containers_healthy; do
  if [ $waited -lt $TIMEOUT_SECONDS ]; then
    sleep $SLEEP_SECONDS
    waited=$((waited + SLEEP_SECONDS))
  else
    echo ""
    echo "ERROR: Timeout after $TIMEOUT_SECONDS seconds."
    echo ""
    echo "The following Docker containers are not healthy yet:"
    docker inspect --format="{{.Name}}:{{.State.Health.Status}}" "${CONTAINER_NAMES[@]}" \
      | sed -r "/:healthy$/d" | sed -rn "s/\/(.*):(.*)/ - \1 (status: \2)/p"
    echo ""
    echo "Please check the container logs."
    echo ""
    exit 1
  fi
done

echo ""
echo "Cawemo is ready! Please open it in your browser at $(get_config_variable "SERVER_URL")."
echo ""
